// Note: In this implementation, malloc() and realloc() correctly manage alignment.
// Recall that all elements in the stack are the same size, so all that is required
// is that the first item be aligned at an appropriate address.

#include <cstdlib>
#include <cassert>
#include <new>
#include "stack.hpp"

using namespace std; // work-around for MSVC 7.0, which doesn't have assert() in std::

namespace {
  const size_t StartCapacity = 10;
  const float GrowthRate = 1.8f;
}

GenericStack::GenericStack(size_t size) :
item_size(size), capacity(size*StartCapacity), nbuf(0)
{
  buf = static_cast<char *>(std::malloc(capacity));
  if (buf == NULL)
    throw std::bad_alloc();
}

GenericStack::~GenericStack()
{
  free(buf);
}

void GenericStack::push()
// Add memory for new item to top
{
  // Allocate more memory, if required
  if (nbuf == capacity) {
    size_t n = static_cast<size_t>(capacity * GrowthRate + 1);
    char *p = static_cast<char *>(std::realloc(buf, n));
    if (p == NULL)
      throw std::bad_alloc();

    buf = p;
    capacity = n;
  }

  // We have sufficient memory, increment stack counter
  nbuf += item_size;
}

void GenericStack::pop() throw()
// Release memory of top item, but does not actually free memory
{
  assert(nbuf > 0);
  nbuf -= item_size;
}

void *GenericStack::top() throw()
// Return pointer to top element
{
  assert(nbuf > 0);
  return buf + nbuf - item_size;
}

bool GenericStack::empty() const throw()
{
  return nbuf == 0;
}

