/* DEMO: Generic interface classes.
 *
 * The following code is an extension of the idea presented in Item 42 of
 * Scott Meyer's "Effective C++". In his discussion he creates a generic
 * stack class to contain pointers-to-objects. The class is written with a
 * void* based implementation, and a template-based interface.
 *
 * The chief aim of restricting templates to interface classes is to reduce
 * "code bloat", which is the replication of template code for each instantiated
 * type. Bloat increases the overall size of the object code.
 *
 * In the demonstration below, I extend the interface-class idea to create
 * a generic stack class to contain objects (ie, not pointers to objects). The
 * same basic approach is applied:
 *  1. Write a void* class to do most of the work.
 *  2. Write a slim template interface class for type-safety etc.
 *
 * A implementation that contains objects (and not just pointers) requires the
 * use of "placement new" and explicit destructor calls. Discussions of these
 * operations can be found in:
 *  1. Stroustrup (Special edition 2000) p 256.
 *  2. Items 22, 23, 28, and 68 of Herb Sutter's "Guru of the Week" website.
 *
 * Tim Bailey 2004.
 */

class GenericStack {
protected:
  GenericStack(size_t size);
  ~GenericStack();

  void push();
  void pop() throw();
  void *top() throw();
  bool empty() const throw();

private:
  GenericStack(const GenericStack&); // prevent copy
  void operator=(const GenericStack&); // prevent assignment

  const size_t item_size;
  size_t capacity;
  size_t nbuf;
  char *buf;
};

template <typename T>
class Stack : private GenericStack {
public:
  Stack() : GenericStack(sizeof(T)) {}
  ~Stack() { while (!empty()) (void)pop(); }

  void push(const T &item) {
    // Allocate memory, then copy-construct with placement new
    GenericStack::push();
    try { new (GenericStack::top()) T(item); }
    catch (...) { GenericStack::pop(); throw; }
  }

  T pop() {
    // Copy stored variable, then explicitly destruct, then release memory
    T *stored = static_cast<T*>(GenericStack::top());
    T t(*stored);
    stored->~T();
    GenericStack::pop();
    return t; // note, this return is not "strong" exception-safe if T's copy constructor can throw.
  }

  bool empty() const throw() { return GenericStack::empty(); }
};

/* NOTES:

  1. To make this Stack<T> fully exception-safe (ie, to provide a "strong
  guarantee" about its state), we need to replace
    T pop()

  with two functions:
    T &top();  - which returns a reference to the top element
    void pop(); - which removes the top element

  For a good discussion on this issue and more, see:
    Kevlin Henney, "From Mechanism to Method: The Safe Stacking of Cats",
    in the C/C++ Users Journal, February, 2002.

  2. A preferable alternative to the try/catch block above is to replace it
  with the "resource acquisition is initialization" idiom, which relies on the
  interaction of constructors/destructors and exception handling.

  A good example of this idiom in the context of our application is the
  dismissable "scope guard" as described in:
    Andrei Alexandrescu, "Generic<Programming>: Change the Way You Write
    Exception-Safe Code  Forever", in the C/C++ Users Journal, December, 2000.
*/

